<?php
/**
 * Settings page template
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

$api_key = get_option( 'pensteady_api_key', '' );
$auto_publish = get_option( 'pensteady_auto_publish', false );
$default_category = get_option( 'pensteady_default_category', 1 );
$default_author = get_option( 'pensteady_default_author', 1 );
$sync_frequency = get_option( 'pensteady_sync_frequency', 'daily' );

$categories = get_categories( array( 'hide_empty' => false ) );
$authors = get_users( array( 'who' => 'authors' ) );
$logs = Pensteady_Cron::get_logs( 20 );
?>

<div class="wrap pensteady-settings">
    <h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
    
    <div class="pensteady-container">
        <div class="pensteady-main">
            <div class="card">
                <h2><?php _e( 'API Configuration', 'pensteady' ); ?></h2>
                
                <form method="post" action="">
                    <?php wp_nonce_field( 'pensteady_settings', 'pensteady_settings_nonce' ); ?>
                    
                    <table class="form-table">
                        <tr>
                            <th scope="row">
                                <label for="pensteady_api_key"><?php _e( 'API Key', 'pensteady' ); ?></label>
                            </th>
                            <td>
                                <input 
                                    type="text" 
                                    id="pensteady_api_key" 
                                    name="pensteady_api_key" 
                                    value="<?php echo esc_attr( $api_key ); ?>" 
                                    class="regular-text"
                                    placeholder="ps_live_..."
                                >
                                <button type="button" class="button button-secondary" id="pensteady-test-connection">
                                    <?php _e( 'Test Connection', 'pensteady' ); ?>
                                </button>
                                <span id="pensteady-test-result"></span>
                                <p class="description">
                                    <?php _e( 'Enter your Pensteady API key (starts with ps_live_)', 'pensteady' ); ?>
                                </p>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="pensteady_auto_publish"><?php _e( 'Auto-publish', 'pensteady' ); ?></label>
                            </th>
                            <td>
                                <label>
                                    <input 
                                        type="checkbox" 
                                        id="pensteady_auto_publish" 
                                        name="pensteady_auto_publish" 
                                        value="1"
                                        <?php checked( $auto_publish ); ?>
                                    >
                                    <?php _e( 'Automatically publish synced articles', 'pensteady' ); ?>
                                </label>
                                <p class="description">
                                    <?php _e( 'If unchecked, articles will be saved as drafts', 'pensteady' ); ?>
                                </p>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="pensteady_default_category"><?php _e( 'Default Category', 'pensteady' ); ?></label>
                            </th>
                            <td>
                                <select id="pensteady_default_category" name="pensteady_default_category">
                                    <?php foreach ( $categories as $category ) : ?>
                                        <option value="<?php echo esc_attr( $category->term_id ); ?>" <?php selected( $default_category, $category->term_id ); ?>>
                                            <?php echo esc_html( $category->name ); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                                <p class="description">
                                    <?php _e( 'Fallback category for synced articles', 'pensteady' ); ?>
                                </p>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="pensteady_default_author"><?php _e( 'Default Author', 'pensteady' ); ?></label>
                            </th>
                            <td>
                                <select id="pensteady_default_author" name="pensteady_default_author">
                                    <?php foreach ( $authors as $author ) : ?>
                                        <option value="<?php echo esc_attr( $author->ID ); ?>" <?php selected( $default_author, $author->ID ); ?>>
                                            <?php echo esc_html( $author->display_name ); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                                <p class="description">
                                    <?php _e( 'Author assigned to synced articles', 'pensteady' ); ?>
                                </p>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="pensteady_sync_frequency"><?php _e( 'Sync Frequency', 'pensteady' ); ?></label>
                            </th>
                            <td>
                                <select id="pensteady_sync_frequency" name="pensteady_sync_frequency">
                                    <option value="hourly" <?php selected( $sync_frequency, 'hourly' ); ?>>
                                        <?php _e( 'Hourly', 'pensteady' ); ?>
                                    </option>
                                    <option value="twice_daily" <?php selected( $sync_frequency, 'twice_daily' ); ?>>
                                        <?php _e( 'Twice Daily', 'pensteady' ); ?>
                                    </option>
                                    <option value="daily" <?php selected( $sync_frequency, 'daily' ); ?>>
                                        <?php _e( 'Daily', 'pensteady' ); ?>
                                    </option>
                                </select>
                                <p class="description">
                                    <?php _e( 'How often to check for new articles', 'pensteady' ); ?>
                                </p>
                            </td>
                        </tr>
                    </table>
                    
                    <p class="submit">
                        <?php submit_button( __( 'Save Settings', 'pensteady' ), 'primary', 'submit', false ); ?>
                        <button type="button" class="button button-secondary" id="pensteady-manual-sync">
                            <?php _e( 'Sync Now', 'pensteady' ); ?>
                        </button>
                        <span id="pensteady-sync-result"></span>
                    </p>
                </form>
            </div>
            
            <div class="card">
                <h2><?php _e( 'Activity Log', 'pensteady' ); ?></h2>
                
                <?php if ( empty( $logs ) ) : ?>
                    <p><?php _e( 'No activity yet.', 'pensteady' ); ?></p>
                <?php else : ?>
                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <th><?php _e( 'Date', 'pensteady' ); ?></th>
                                <th><?php _e( 'Article', 'pensteady' ); ?></th>
                                <th><?php _e( 'Status', 'pensteady' ); ?></th>
                                <th><?php _e( 'Message', 'pensteady' ); ?></th>
                                <th><?php _e( 'Post', 'pensteady' ); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ( $logs as $log ) : ?>
                                <tr>
                                    <td><?php echo esc_html( mysql2date( 'Y-m-d H:i:s', $log->created_at ) ); ?></td>
                                    <td><?php echo esc_html( $log->article_title ?: $log->article_id ); ?></td>
                                    <td>
                                        <span class="pensteady-status pensteady-status-<?php echo esc_attr( $log->status ); ?>">
                                            <?php echo esc_html( ucfirst( $log->status ) ); ?>
                                        </span>
                                    </td>
                                    <td><?php echo esc_html( $log->message ); ?></td>
                                    <td>
                                        <?php if ( $log->post_id ) : ?>
                                            <a href="<?php echo esc_url( get_edit_post_link( $log->post_id ) ); ?>" target="_blank">
                                                #<?php echo esc_html( $log->post_id ); ?>
                                            </a>
                                        <?php else : ?>
                                            —
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            </div>
        </div>
        
        <div class="pensteady-sidebar">
            <div class="card">
                <h3><?php _e( 'About Pensteady', 'pensteady' ); ?></h3>
                <p><?php _e( 'Pensteady automatically syncs scheduled articles from your Pensteady account to WordPress.', 'pensteady' ); ?></p>
                <p>
                    <a href="https://pensteady.com" target="_blank" class="button button-secondary">
                        <?php _e( 'Visit Pensteady', 'pensteady' ); ?>
                    </a>
                </p>
            </div>
            
            <div class="card">
                <h3><?php _e( 'Need Help?', 'pensteady' ); ?></h3>
                <ul>
                    <li><a href="https://pensteady.com/docs" target="_blank"><?php _e( 'Documentation', 'pensteady' ); ?></a></li>
                    <li><a href="https://pensteady.com/support" target="_blank"><?php _e( 'Support', 'pensteady' ); ?></a></li>
                </ul>
            </div>
        </div>
    </div>
</div>

<script>
jQuery(document).ready(function($) {
    // Test connection
    $('#pensteady-test-connection').on('click', function() {
        var $button = $(this);
        var $result = $('#pensteady-test-result');
        var apiKey = $('#pensteady_api_key').val();
        
        if (!apiKey) {
            $result.html('<span class="error">⚠️ Please enter an API key</span>');
            return;
        }
        
        $button.prop('disabled', true).text('<?php _e( 'Testing...', 'pensteady' ); ?>');
        $result.html('<span class="pending">⏳</span>');
        
        $.post(pensteady.ajax_url, {
            action: 'pensteady_test_connection',
            nonce: pensteady.nonce,
            api_key: apiKey
        }, function(response) {
            if (response.success) {
                $result.html('<span class="success">✅ ' + response.data.message + '</span>');
            } else {
                $result.html('<span class="error">❌ ' + response.data.message + '</span>');
            }
        }).fail(function() {
            $result.html('<span class="error">❌ Connection failed</span>');
        }).always(function() {
            $button.prop('disabled', false).text('<?php _e( 'Test Connection', 'pensteady' ); ?>');
        });
    });
    
    // Manual sync
    $('#pensteady-manual-sync').on('click', function() {
        var $button = $(this);
        var $result = $('#pensteady-sync-result');
        
        $button.prop('disabled', true).text('<?php _e( 'Syncing...', 'pensteady' ); ?>');
        $result.html('<span class="pending">⏳</span>');
        
        $.post(pensteady.ajax_url, {
            action: 'pensteady_manual_sync',
            nonce: pensteady.nonce
        }, function(response) {
            if (response.success) {
                $result.html('<span class="success">✅ ' + response.data.message + '</span>');
                setTimeout(function() {
                    location.reload();
                }, 1500);
            } else {
                $result.html('<span class="error">❌ ' + response.data.message + '</span>');
            }
        }).fail(function() {
            $result.html('<span class="error">❌ Sync failed</span>');
        }).always(function() {
            $button.prop('disabled', false).text('<?php _e( 'Sync Now', 'pensteady' ); ?>');
        });
    });
});
</script>
