<?php
/**
 * Pensteady Admin Interface
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Pensteady_Admin {
    
    /**
     * Constructor
     */
    public function __construct() {
        add_action( 'admin_menu', array( $this, 'add_settings_page' ) );
        add_action( 'admin_init', array( $this, 'register_settings' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );
        add_action( 'wp_ajax_pensteady_test_connection', array( $this, 'ajax_test_connection' ) );
        add_action( 'wp_ajax_pensteady_manual_sync', array( $this, 'ajax_manual_sync' ) );
    }
    
    /**
     * Add settings page
     */
    public function add_settings_page() {
        add_options_page(
            __( 'Pensteady Settings', 'pensteady' ),
            __( 'Pensteady', 'pensteady' ),
            'manage_options',
            'pensteady',
            array( $this, 'render_settings_page' )
        );
    }
    
    /**
     * Register settings
     */
    public function register_settings() {
        register_setting( 'pensteady_settings', 'pensteady_api_key', array(
            'type'              => 'string',
            'sanitize_callback' => 'sanitize_text_field',
        ) );
        
        register_setting( 'pensteady_settings', 'pensteady_auto_publish', array(
            'type'              => 'boolean',
            'default'           => false,
            'sanitize_callback' => 'rest_sanitize_boolean',
        ) );
        
        register_setting( 'pensteady_settings', 'pensteady_default_category', array(
            'type'              => 'integer',
            'default'           => 1,
            'sanitize_callback' => 'absint',
        ) );
        
        register_setting( 'pensteady_settings', 'pensteady_default_author', array(
            'type'              => 'integer',
            'default'           => 1,
            'sanitize_callback' => 'absint',
        ) );
        
        register_setting( 'pensteady_settings', 'pensteady_sync_frequency', array(
            'type'              => 'string',
            'default'           => 'daily',
            'sanitize_callback' => array( $this, 'sanitize_frequency' ),
        ) );
    }
    
    /**
     * Sanitize frequency value
     */
    public function sanitize_frequency( $value ) {
        $allowed = array( 'hourly', 'twice_daily', 'daily' );
        return in_array( $value, $allowed, true ) ? $value : 'daily';
    }
    
    /**
     * Enqueue admin assets
     */
    public function enqueue_admin_assets( $hook ) {
        if ( 'settings_page_pensteady' !== $hook ) {
            return;
        }
        
        wp_enqueue_style(
            'pensteady-admin',
            PENSTEADY_PLUGIN_URL . 'admin/css/admin.css',
            array(),
            PENSTEADY_VERSION
        );
        
        wp_enqueue_script(
            'pensteady-admin',
            PENSTEADY_PLUGIN_URL . 'admin/js/admin.js',
            array( 'jquery' ),
            PENSTEADY_VERSION,
            true
        );
        
        wp_localize_script( 'pensteady-admin', 'pensteady', array(
            'ajax_url' => admin_url( 'admin-ajax.php' ),
            'nonce'    => wp_create_nonce( 'pensteady_ajax' ),
        ) );
    }
    
    /**
     * Render settings page
     */
    public function render_settings_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        
        // Handle settings update
        if ( isset( $_POST['pensteady_settings_nonce'] ) && 
             wp_verify_nonce( $_POST['pensteady_settings_nonce'], 'pensteady_settings' ) ) {
            
            update_option( 'pensteady_api_key', sanitize_text_field( $_POST['pensteady_api_key'] ?? '' ) );
            update_option( 'pensteady_auto_publish', isset( $_POST['pensteady_auto_publish'] ) );
            update_option( 'pensteady_default_category', absint( $_POST['pensteady_default_category'] ?? 1 ) );
            update_option( 'pensteady_default_author', absint( $_POST['pensteady_default_author'] ?? 1 ) );
            
            $old_frequency = get_option( 'pensteady_sync_frequency', 'daily' );
            $new_frequency = $this->sanitize_frequency( $_POST['pensteady_sync_frequency'] ?? 'daily' );
            update_option( 'pensteady_sync_frequency', $new_frequency );
            
            // Reschedule cron if frequency changed
            if ( $old_frequency !== $new_frequency ) {
                $timestamp = wp_next_scheduled( 'pensteady_sync_articles' );
                if ( $timestamp ) {
                    wp_unschedule_event( $timestamp, 'pensteady_sync_articles' );
                }
                wp_schedule_event( time(), $new_frequency, 'pensteady_sync_articles' );
            }
            
            echo '<div class="notice notice-success"><p>' . __( 'Settings saved.', 'pensteady' ) . '</p></div>';
        }
        
        include PENSTEADY_PLUGIN_DIR . 'admin/settings-page.php';
    }
    
    /**
     * AJAX: Test connection
     */
    public function ajax_test_connection() {
        check_ajax_referer( 'pensteady_ajax', 'nonce' );
        
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Permission denied', 'pensteady' ) ) );
        }
        
        $api_key = sanitize_text_field( $_POST['api_key'] ?? '' );
        
        if ( empty( $api_key ) ) {
            wp_send_json_error( array( 'message' => __( 'API key is required', 'pensteady' ) ) );
        }
        
        $api = new Pensteady_API( $api_key );
        $result = $api->test_connection();
        
        if ( $result['success'] ) {
            wp_send_json_success( $result );
        } else {
            wp_send_json_error( $result );
        }
    }
    
    /**
     * AJAX: Manual sync
     */
    public function ajax_manual_sync() {
        check_ajax_referer( 'pensteady_ajax', 'nonce' );
        
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Permission denied', 'pensteady' ) ) );
        }
        
        $cron = new Pensteady_Cron();
        $result = $cron->sync_articles();
        
        wp_send_json_success( array(
            'message' => sprintf(
                __( 'Sync complete. %d articles processed.', 'pensteady' ),
                $result['processed'] ?? 0
            ),
            'result' => $result
        ) );
    }
}
