<?php
/**
 * Pensteady API Client
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Pensteady_API {
    
    /**
     * API base URL
     */
    private $api_base = 'https://pensteady.com/api';
    
    /**
     * API key
     */
    private $api_key;
    
    /**
     * Constructor
     */
    public function __construct( $api_key = null ) {
        $this->api_key = $api_key ?: get_option( 'pensteady_api_key' );
    }
    
    /**
     * Test API connection
     */
    public function test_connection() {
        $response = $this->get( '/articles', array( 'limit' => 1 ) );
        
        if ( is_wp_error( $response ) ) {
            return array(
                'success' => false,
                'message' => $response->get_error_message()
            );
        }
        
        return array(
            'success' => true,
            'message' => __( 'Connection successful!', 'pensteady' )
        );
    }
    
    /**
     * Get scheduled articles
     */
    public function get_scheduled_articles() {
        return $this->get( '/articles', array( 'status' => 'scheduled' ) );
    }
    
    /**
     * Mark article as published
     */
    public function mark_published( $article_id ) {
        return $this->post( '/articles/' . $article_id . '/publish', array() );
    }
    
    /**
     * GET request
     */
    private function get( $endpoint, $params = array() ) {
        $url = $this->api_base . $endpoint;
        
        if ( ! empty( $params ) ) {
            $url = add_query_arg( $params, $url );
        }
        
        $response = wp_remote_get( $url, array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $this->api_key,
                'Content-Type'  => 'application/json',
            ),
            'timeout' => 30,
        ) );
        
        return $this->handle_response( $response );
    }
    
    /**
     * POST request
     */
    private function post( $endpoint, $data = array() ) {
        $url = $this->api_base . $endpoint;
        
        $response = wp_remote_post( $url, array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $this->api_key,
                'Content-Type'  => 'application/json',
            ),
            'body'    => json_encode( $data ),
            'timeout' => 30,
        ) );
        
        return $this->handle_response( $response );
    }
    
    /**
     * Handle API response
     */
    private function handle_response( $response ) {
        if ( is_wp_error( $response ) ) {
            return $response;
        }
        
        $code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );
        
        if ( $code < 200 || $code >= 300 ) {
            $message = sprintf( __( 'API Error: HTTP %d', 'pensteady' ), $code );
            
            $decoded = json_decode( $body, true );
            if ( isset( $decoded['error'] ) ) {
                $message .= ' - ' . $decoded['error'];
            } elseif ( isset( $decoded['message'] ) ) {
                $message .= ' - ' . $decoded['message'];
            }
            
            return new WP_Error( 'pensteady_api_error', $message );
        }
        
        $decoded = json_decode( $body, true );
        
        if ( json_last_error() !== JSON_ERROR_NONE ) {
            return new WP_Error( 'pensteady_json_error', __( 'Invalid JSON response', 'pensteady' ) );
        }
        
        return $decoded;
    }
}
