<?php
/**
 * Pensteady Cron Handler
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Pensteady_Cron {
    
    /**
     * Constructor
     */
    public function __construct() {
        add_action( 'pensteady_sync_articles', array( $this, 'sync_articles' ) );
    }
    
    /**
     * Sync articles from Pensteady
     */
    public function sync_articles() {
        $api_key = get_option( 'pensteady_api_key' );
        
        if ( empty( $api_key ) ) {
            $this->log( null, null, 'error', 'No API key configured' );
            return array( 'processed' => 0, 'errors' => array( 'No API key configured' ) );
        }
        
        $api = new Pensteady_API( $api_key );
        $response = $api->get_scheduled_articles();
        
        if ( is_wp_error( $response ) ) {
            $this->log( null, null, 'error', $response->get_error_message() );
            return array( 'processed' => 0, 'errors' => array( $response->get_error_message() ) );
        }
        
        $articles = $response['data'] ?? $response['articles'] ?? array();
        
        if ( empty( $articles ) ) {
            $this->log( null, null, 'info', 'No scheduled articles found' );
            return array( 'processed' => 0, 'errors' => array() );
        }
        
        $auto_publish = get_option( 'pensteady_auto_publish', false );
        $default_category = get_option( 'pensteady_default_category', 1 );
        $default_author = get_option( 'pensteady_default_author', 1 );
        
        $processed = 0;
        $errors = array();
        
        foreach ( $articles as $article ) {
            try {
                $result = $this->create_post( $article, $auto_publish, $default_category, $default_author );
                
                if ( is_wp_error( $result ) ) {
                    $errors[] = $result->get_error_message();
                    $this->log(
                        $article['id'] ?? '',
                        $article['title'] ?? '',
                        'error',
                        $result->get_error_message()
                    );
                } else {
                    $processed++;
                    
                    // Mark as published in Pensteady
                    $mark_result = $api->mark_published( $article['id'] );
                    
                    if ( is_wp_error( $mark_result ) ) {
                        $this->log(
                            $article['id'],
                            $article['title'],
                            'warning',
                            'Post created but failed to mark as published in Pensteady: ' . $mark_result->get_error_message(),
                            $result
                        );
                    } else {
                        $this->log(
                            $article['id'],
                            $article['title'],
                            'success',
                            'Successfully synced',
                            $result
                        );
                    }
                }
            } catch ( Exception $e ) {
                $errors[] = $e->getMessage();
                $this->log(
                    $article['id'] ?? '',
                    $article['title'] ?? '',
                    'error',
                    $e->getMessage()
                );
            }
        }
        
        return array(
            'processed' => $processed,
            'errors'    => $errors,
        );
    }
    
    /**
     * Create WordPress post from article
     */
    private function create_post( $article, $auto_publish, $default_category, $default_author ) {
        // Check if post already exists
        global $wpdb;
        $existing_post = $wpdb->get_var( $wpdb->prepare(
            "SELECT post_id FROM {$wpdb->prefix}pensteady_log WHERE article_id = %s AND status = 'success' LIMIT 1",
            $article['id']
        ) );
        
        if ( $existing_post ) {
            return new WP_Error( 'duplicate', sprintf(
                __( 'Post already exists (ID: %d)', 'pensteady' ),
                $existing_post
            ) );
        }
        
        // Prepare post data
        $post_data = array(
            'post_title'   => wp_strip_all_tags( $article['title'] ?? '' ),
            'post_content' => $article['content'] ?? '',
            'post_excerpt' => $article['excerpt'] ?? '',
            'post_status'  => $auto_publish ? 'publish' : 'draft',
            'post_author'  => $default_author,
            'post_type'    => 'post',
        );
        
        // Insert post
        $post_id = wp_insert_post( $post_data, true );
        
        if ( is_wp_error( $post_id ) ) {
            return $post_id;
        }
        
        // Set category
        if ( ! empty( $default_category ) ) {
            wp_set_post_categories( $post_id, array( $default_category ) );
        }
        
        // Handle categories from article
        if ( ! empty( $article['categories'] ) ) {
            $category_ids = array();
            foreach ( $article['categories'] as $cat_name ) {
                $cat = get_category_by_slug( sanitize_title( $cat_name ) );
                if ( ! $cat ) {
                    $cat_id = wp_create_category( $cat_name );
                } else {
                    $cat_id = $cat->term_id;
                }
                if ( $cat_id ) {
                    $category_ids[] = $cat_id;
                }
            }
            if ( ! empty( $category_ids ) ) {
                wp_set_post_categories( $post_id, $category_ids );
            }
        }
        
        // Handle tags
        if ( ! empty( $article['tags'] ) ) {
            wp_set_post_tags( $post_id, $article['tags'], false );
        }
        
        // Handle featured image
        if ( ! empty( $article['featured_image'] ) ) {
            $this->set_featured_image( $post_id, $article['featured_image'] );
        }
        
        // Store article ID in post meta
        update_post_meta( $post_id, '_pensteady_article_id', $article['id'] );
        
        return $post_id;
    }
    
    /**
     * Set featured image from URL
     */
    private function set_featured_image( $post_id, $image_url ) {
        require_once( ABSPATH . 'wp-admin/includes/image.php' );
        require_once( ABSPATH . 'wp-admin/includes/file.php' );
        require_once( ABSPATH . 'wp-admin/includes/media.php' );
        
        // Download image
        $tmp = download_url( $image_url );
        
        if ( is_wp_error( $tmp ) ) {
            return $tmp;
        }
        
        // Get file extension
        $file_array = array(
            'name'     => basename( $image_url ),
            'tmp_name' => $tmp,
        );
        
        // Upload to media library
        $attachment_id = media_handle_sideload( $file_array, $post_id );
        
        if ( is_wp_error( $attachment_id ) ) {
            @unlink( $file_array['tmp_name'] );
            return $attachment_id;
        }
        
        // Set as featured image
        set_post_thumbnail( $post_id, $attachment_id );
        
        return $attachment_id;
    }
    
    /**
     * Log activity
     */
    private function log( $article_id, $article_title, $status, $message, $post_id = null ) {
        global $wpdb;
        
        $wpdb->insert(
            $wpdb->prefix . 'pensteady_log',
            array(
                'article_id'    => $article_id,
                'article_title' => $article_title,
                'post_id'       => $post_id,
                'status'        => $status,
                'message'       => $message,
                'created_at'    => current_time( 'mysql' ),
            ),
            array( '%s', '%s', '%d', '%s', '%s', '%s' )
        );
    }
    
    /**
     * Get recent logs
     */
    public static function get_logs( $limit = 20 ) {
        global $wpdb;
        
        return $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM {$wpdb->prefix}pensteady_log ORDER BY created_at DESC LIMIT %d",
                $limit
            )
        );
    }
}
