<?php
/**
 * Plugin Name: Pensteady
 * Plugin URI: https://pensteady.com
 * Description: Automatically sync scheduled articles from Pensteady to WordPress
 * Version: 1.0.0
 * Author: Pensteady
 * Author URI: https://pensteady.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: pensteady
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Define plugin constants
define( 'PENSTEADY_VERSION', '1.0.0' );
define( 'PENSTEADY_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'PENSTEADY_PLUGIN_URL', plugin_dir_url( __FILE__ ) );

// Include required files
require_once PENSTEADY_PLUGIN_DIR . 'includes/class-pensteady-api.php';
require_once PENSTEADY_PLUGIN_DIR . 'includes/class-pensteady-admin.php';
require_once PENSTEADY_PLUGIN_DIR . 'includes/class-pensteady-cron.php';

/**
 * Initialize the plugin
 */
function pensteady_init() {
    // Initialize admin interface
    if ( is_admin() ) {
        new Pensteady_Admin();
    }
    
    // Initialize cron functionality
    new Pensteady_Cron();
}
add_action( 'plugins_loaded', 'pensteady_init' );

/**
 * Activation hook
 */
function pensteady_activate() {
    // Create activity log table
    global $wpdb;
    $table_name = $wpdb->prefix . 'pensteady_log';
    $charset_collate = $wpdb->get_charset_collate();
    
    $sql = "CREATE TABLE IF NOT EXISTS $table_name (
        id mediumint(9) NOT NULL AUTO_INCREMENT,
        article_id varchar(255) NOT NULL,
        article_title text NOT NULL,
        post_id bigint(20),
        status varchar(50) NOT NULL,
        message text,
        created_at datetime DEFAULT CURRENT_TIMESTAMP NOT NULL,
        PRIMARY KEY  (id),
        KEY article_id (article_id),
        KEY created_at (created_at)
    ) $charset_collate;";
    
    require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
    dbDelta( $sql );
    
    // Schedule cron if not already scheduled
    if ( ! wp_next_scheduled( 'pensteady_sync_articles' ) ) {
        $frequency = get_option( 'pensteady_sync_frequency', 'daily' );
        wp_schedule_event( time(), $frequency, 'pensteady_sync_articles' );
    }
}
register_activation_hook( __FILE__, 'pensteady_activate' );

/**
 * Deactivation hook
 */
function pensteady_deactivate() {
    // Clear scheduled cron
    $timestamp = wp_next_scheduled( 'pensteady_sync_articles' );
    if ( $timestamp ) {
        wp_unschedule_event( $timestamp, 'pensteady_sync_articles' );
    }
}
register_deactivation_hook( __FILE__, 'pensteady_deactivate' );

/**
 * Add custom cron schedules
 */
function pensteady_cron_schedules( $schedules ) {
    $schedules['twice_daily'] = array(
        'interval' => 12 * HOUR_IN_SECONDS,
        'display'  => __( 'Twice Daily', 'pensteady' )
    );
    return $schedules;
}
add_filter( 'cron_schedules', 'pensteady_cron_schedules' );
